/**
 * 
 */
package be.SIRAPRISE.util;

import java.util.HashMap;
import java.util.Properties;
import java.util.Map.Entry;

import be.erwinsmout.MyProperties;

/**
 * @author Erwin
 * 
 */
public class CommandLineParameters extends Properties {

	/**
	 * 
	 */
	private static final long serialVersionUID = 3300117546127188106L;

	/**
	 * Gets a parameter map representing the contents of the command line parameters.
	 * <P>
	 * Command line parameters can be specified using two syntaxes :
	 * <ol>
	 * <li>optionName(optionValue)</LI>
	 * <li>-optionName:OptionValue</LI>
	 * </ol>
	 * </P>
	 * 
	 * <P>
	 * There is one option that can be declared the default option. The value for this option can be specified without specifying the option name. E.g. if the default option is FILE, then the following command lines are equivalent :
	 * <ol>
	 * <li>java JavaMainClass optionValue</LI>
	 * <li>java JavaMainClass FILE(optionValue)</LI>
	 * <li>java JavaMainClass -FILE:optionValue</LI>
	 * </ol>
	 * </P>
	 * 
	 * @param args
	 *            The command line parameters in String[] format
	 * @param defaultPropertyName
	 *            The name of the property that must not be typed, or null if there is no default property
	 * @return A parameter map representing the contents of the command line parameters
	 */
	private static HashMap<String, String> getCommandLineParameters (String[] args, String defaultPropertyName) {
		HashMap<String, String> commandLineParametersMap = new HashMap<String, String>();
		int i = 0, argc = args.length;
		while (i < argc) {
			String arg = args[i];
			try {
				NameValuePair nameValuePair = BracketParser.getNameAndMandatoryValueFromNonEscaped(arg, 0).getNameValuePair();
				String argumentName = nameValuePair.getName();
				String argumentValue = nameValuePair.getValue();

				if (argumentValue.length() < 1) {
					if (defaultPropertyName == null) {
						throw new IllegalArgumentException(arg);
					}
					commandLineParametersMap.put(defaultPropertyName.toUpperCase(), MyProperties.resolveSystemPropertyRefs(arg));
				} else {
					commandLineParametersMap.put(argumentName, MyProperties.resolveSystemPropertyRefs(argumentValue));
				}
			} catch (NoClosingBracketException e) {
				throw new IllegalArgumentException(arg);
			} catch (NoOpeningBracketException e) {
				// See if it is a "-option:value" structure
				if (arg.charAt(0) == '-') {
					arg = arg.substring(1);
					int c = arg.indexOf(':');
					if (c < 1) {
						throw new IllegalArgumentException(arg);
					}
					commandLineParametersMap.put(arg.substring(0, c).toUpperCase(), MyProperties.resolveSystemPropertyRefs(arg.substring(c + 1)));
				} else {
					if (defaultPropertyName == null) {
						throw new IllegalArgumentException(arg);
					}
					commandLineParametersMap.put(defaultPropertyName.toUpperCase(), MyProperties.resolveSystemPropertyRefs(arg));
				}
			}
			i++;
		}
		return commandLineParametersMap;
	}

	/**
	 * Gets a Properties object representing the contents of the command line parameters, with the defaults for the specifyable options held in the given defaultProperties.
	 * <P>
	 * Command line parameters can be specified using two syntaxes :
	 * <ol>
	 * <li>optionName(optionValue)</LI>
	 * <li>-optionName:OptionValue</LI>
	 * </OL>
	 * </P>
	 * 
	 * <P>
	 * There is one option that can be declared the default option. The value for this option can be specified without specifying the option name. E.g. if the default option is FILE, then the following command lines are equivalent :
	 * <ol>
	 * <li>java JavaMainClass optionValue</LI>
	 * <li>java JavaMainClass FILE(optionValue)</LI>
	 * <li>java JavaMainClass -FILE:optionValue</LI>
	 * </OL>
	 * </P>
	 * 
	 * 
	 * @param args
	 *            The command line parameters in String[] format
	 * @param defaultPropertyName
	 *            The name of the property that must not be typed, or null if there is no default property
	 * @param defaultProperties
	 *            The Properties object containing the defaults for the options specifyable through the command line, or null if there are no defaults
	 * @return A parameter map representing the contents of the command line parameters
	 */
	public static CommandLineParameters getCommandLineParameters (String[] args, String defaultPropertyName, Properties defaultProperties) {
		CommandLineParameters commandLineProperties = new CommandLineParameters(defaultProperties);
		for (Entry<String, String> entry : CommandLineParameters.getCommandLineParameters(args, defaultPropertyName).entrySet()) {
			commandLineProperties.setProperty(entry.getKey(), entry.getValue());
		}
		return commandLineProperties;
	}

	/**
	 * Creates the CommandLineParameters
	 * 
	 * @param defaultProperties
	 *            The Properties object containing the defaults for the options specifyable through the command line, or null if there are no defaults
	 */
	private CommandLineParameters (Properties defaultProperties) {
		super(defaultProperties);
	}
}
