/*
 * Created on 27-jun-2005
 */
package be.SIRAPRISE.util;

import java.util.*;

/**
 * Defines the intersect, minus and union operations on maps
 * 
 * @author Erwin Smout
 * @param <K>
 * @param <V>
 */
public abstract class IntersectableMap<K, V> implements Map<K, V> {

	/**
	 * Gets The actual map
	 * 
	 * @return The actual map
	 */
	abstract Map<K, V> getMap ( );

	/**
	 * Gets a clone of the inner map
	 * 
	 * @return a clone of the inner map
	 */
	abstract Map<K, V> getMapClone ( );

	/**
	 * Sets the inner map object in the context of cloning
	 * 
	 * @param map
	 *            The inner map object
	 */
	abstract void setMap (Map<K, V> map);

	public final void clear ( ) {
		getMap().clear();
	}

	@SuppressWarnings("unchecked")
	public final Object clone ( ) {
		IntersectableMap<K, V> newIntersectableMap = null;
		try {
			newIntersectableMap = (IntersectableMap<K, V>) super.clone();
			newIntersectableMap.setMap(getMapClone());
		} catch (CloneNotSupportedException e) {
			// assert false;
		}
		return newIntersectableMap;
	}

	public final boolean containsKey (Object key) {
		return getMap().containsKey(key);
	}

	public final boolean containsValue (Object value) {
		return getMap().containsValue(value);
	}

	public final Set<Entry<K, V>> entrySet ( ) {
		return getMap().entrySet();
	}

	public final boolean equals (Object obj) {
		return obj instanceof IntersectableMap<?, ?> ? getMap().equals(((IntersectableMap<?, ?>) obj).getMap()) : getMap().equals(obj);
	}

	public final V get (Object key) {
		return getMap().get(key);
	}

	public final int hashCode ( ) {
		return getMap().hashCode();
	}

	/**
	 * Returns a Map that holds only the entries whose keys also occur in keys, and whose class is the same as the caller's class.
	 * 
	 * @param keys
	 *            The set that is to be intersected with the keys of this map
	 * @return A map holding this map's entries whose keys also occur in keys
	 */
	public final IntersectableMap<K, V> intersect (Set<?> keys) {
		IntersectableMap<K, V> intersect = this.getClone();

		Iterator<K> i = intersect.keySet().iterator();
		while (i.hasNext()) {
			K k = i.next();
			if (!keys.contains(k)) {
				i.remove();
			}
		}

		return intersect;
	}

	/**
	 * Gets A copy of this map
	 * 
	 * @return A copy of this map
	 */
	protected abstract IntersectableMap<K, V> getClone ( );

	/**
	 * Returns a Map that holds only the entries whose keys also occur in keys
	 * 
	 * @param keys
	 *            The set that is to be intersected with the keys of this map
	 * @param resultClass
	 *            the Class that the result object must be
	 * @return A map holding this map's entries whose keys also occur in keys
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	@SuppressWarnings("unchecked")
	public final Map<K, V> intersect (Set<?> keys, Class<? extends Map<K, V>> resultClass) throws InstantiationException, IllegalAccessException {
		if (!Map.class.isAssignableFrom(resultClass)) {
			throw new IllegalArgumentException();
		}

		Map<K, V> intersect = resultClass.newInstance();

		if (this.size() < keys.size()) {
			Iterator<Entry<K, V>> i = this.entrySet().iterator();
			while (i.hasNext()) {
				Entry<K, V> me = i.next();
				K k = me.getKey();
				if (keys.contains(k)) {
					intersect.put(k, me.getValue());
				}
			}
		} else {
			Iterator<?> i = keys.iterator();
			while (i.hasNext()) {
				Object k = i.next();
				if (containsKey(k)) {
					intersect.put((K) k, get(k));
				}
			}
		}
		return intersect;
	}

	public final boolean isEmpty ( ) {
		return getMap().isEmpty();
	}

	public final Set<K> keySet ( ) {
		return getMap().keySet();
	}

	/**
	 * Returns a Map that holds only the entries whose keys do not occur in set, and whose class is the same as the caller's class.
	 * 
	 * @param keys
	 *            The set that is to be intersected with the keys of this map
	 * @return A map holding this map's entries whose keys also occur in keys
	 */
	public final IntersectableMap<K, V> minus (Set<?> keys) {
		IntersectableMap<K, V> minus = this.getClone();

		Iterator<K> i = minus.keySet().iterator();
		while (i.hasNext()) {
			K k = i.next();
			if (keys.contains(k)) {
				i.remove();
			}
		}

		return minus;
	}

	/**
	 * Returns a Map that holds only the entries whose keys do not occur in keys
	 * 
	 * @param keys
	 *            The set that is to be differentiated with the keys of this map
	 * @param resultClass
	 *            the Class that the result object must be
	 * @return A map holding this map's entries whose keys do not occur in keys
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	public final Map<K, V> minus (Set<?> keys, Class<? extends Map<K, V>> resultClass) throws InstantiationException, IllegalAccessException {
		if (!Map.class.isAssignableFrom(resultClass)) {
			throw new IllegalArgumentException();
		}

		Map<K, V> minus = resultClass.newInstance();

		if (this.size() < 3 * keys.size()) {
			Iterator<Entry<K, V>> i = this.entrySet().iterator();
			while (i.hasNext()) {
				Entry<K, V> me = i.next();
				K k = me.getKey();
				if (!keys.contains(k)) {
					minus.put(k, me.getValue());
				}
			}
		} else {
			minus.putAll(this);
			Iterator<?> i = keys.iterator();
			while (i.hasNext()) {
				Object k = i.next();
				if (containsKey(k)) {
					minus.remove(k);
				}
			}
		}
		return minus;
	}

	public final V put (K key, V value) {
		return getMap().put(key, value);
	}

	public final void putAll (Map<? extends K, ? extends V> t) {
		getMap().putAll(t);
	}

	public final V remove (Object key) {
		return getMap().remove(key);
	}

	public final int size ( ) {
		return getMap().size();
	}

	public String toString ( ) {
		return getMap().toString();
	}

	/**
	 * Returns a Map that holds only the entries whose keys do not occur in set, and whose class is the same as the caller's class.
	 * 
	 * @param otherMap
	 *            the map to be unioned with this one
	 * @return A map holding this map's entries whose keys also occur in keys
	 * @throws EqualKeyUnequalValuesException
	 */
	public final IntersectableMap<K, V> union (Map<K, V> otherMap) throws EqualKeyUnequalValuesException {
		IntersectableMap<K, V> union = this.getClone();

		Iterator<Entry<K, V>> i = otherMap.entrySet().iterator();
		while (i.hasNext()) {
			Entry<K, V> me = i.next();
			K key = me.getKey();
			V value = me.getValue();
			if (union.containsKey(key)) {
				V valueUnion = union.get(key);
				if (!valueUnion.equals(value)) {
					throw new EqualKeyUnequalValuesException(key, value, valueUnion);
				}
			} else {
				union.put(key, value);
			}
		}

		return union;
	}

	/**
	 * Returns a Map that holds the entries that occur in either this or the other map.
	 * 
	 * @param otherMap
	 *            the map to be unioned with this one
	 * @param resultClass
	 *            the Class that the result object must be
	 * @return A map holding all of this and othermap's entries
	 * @throws EqualKeyUnequalValuesException
	 * @throws IllegalAccessException
	 * @throws InstantiationException
	 */
	public final Map<K, V> union (Map<K, V> otherMap, Class<Map<K, V>> resultClass) throws EqualKeyUnequalValuesException, InstantiationException, IllegalAccessException {
		if (!Map.class.isAssignableFrom(resultClass)) {
			throw new IllegalArgumentException();
		}

		Map<K, V> union = resultClass.newInstance();

		Map<K, V> bulkMap;
		Map<K, V> iterateMap;
		if (this.size() > otherMap.size()) {
			bulkMap = this;
			iterateMap = otherMap;
		} else {
			bulkMap = otherMap;
			iterateMap = this;
		}

		union.putAll(bulkMap);
		Iterator<Entry<K, V>> i_iterateMap = iterateMap.entrySet().iterator();
		while (i_iterateMap.hasNext()) {
			Entry<K, V> me = i_iterateMap.next();
			K key = me.getKey();
			V value = me.getValue();
			if (union.containsKey(key)) {
				Object valueUnion = union.get(key);
				if (valueUnion == null) {
					union.put(key, value);
				} else {
					if (!(value == null || valueUnion.equals(value))) {
						throw new EqualKeyUnequalValuesException(key, value, valueUnion);
					}
				}
			} else {
				union.put(key, value);
			}
		}
		return union;
	}

	public final Collection<V> values ( ) {
		return getMap().values();
	}
}