/**
 * 
 */
package be.SIRAPRISE.util;

import java.io.IOException;

import java.io.InputStream;
import java.security.cert.*;

import be.erwinsmout.MyResource;
import be.erwinsmout.NotFoundException;

/**
 * @author Erwin
 * 
 */
public abstract class MyCertificate {

	/**
	 * 
	 */
	public static final String RESOURCETYPE = "certificate"; //$NON-NLS-1$

	/**
	 * 
	 */
	public static final String SUFFIX = '.' + RESOURCETYPE;

	/**
	 * 
	 */
	public static final String X509CERTIFICATETYPE = "X.509"; //$NON-NLS-1$

	/**
	 * Gets the base name for the certificate resource corresponding to the given class. This is as follows (first rule that applies) :
	 * <OL>
	 * <LI>full classname does not end with 'CERTIFICATE' (case insensitive) ===> full classname, with either of dots and forward slashes as separators, suffixed with '.CERTIFICATE' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * <LI>full classname ends with '.CERTIFICATE', or is equal to 'CERTIFICATE' (both case insensitive) ===> full classname, with either of dots and forward slashes as separators (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties", or just "properties")</LI>
	 * <LI>full classname ends with 'CERTIFICATE', not preceded by a dot (i.e. X.Y.ZCERTIFICATE) ===> X.Y.Z, with either of dots and forward slashes as separators, suffixed with '.CERTIFICATE' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            the class for which the corresponding properties resource name is to be returned
	 * @return the base name for the properties resource corresponding to the given class.
	 */
	protected static String getCertificateBaseName (Class<?> clazz) {
		return MyResource.getResourceBaseName(clazz, RESOURCETYPE);
	}

	/**
	 * Gets a Certificate object of the default certificate type.
	 * 
	 * @param clazz
	 *            The class of the object for which a certificate is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @return The Certificate object containing the contents of the certificate resource corresponding to the clazz class/object
	 * @throws CertificateException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws CertificateException
	 */
	public static Certificate getCertificate (Class<?> clazz) throws CertificateException, NotFoundException, IOException, CertificateException {
		return getCertificate(clazz, X509CERTIFICATETYPE);
	}

	/**
	 * Gets a Certificate object of the given certificate type.
	 * 
	 * @param clazz
	 *            The class of the object for which a certificate is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @param certificateType
	 *            The name of the Certificate type to be returned to the caller
	 * @return The Certificate object containing the contents of the certificate resource corresponding to the clazz class/object
	 * @throws CertificateException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws CertificateException
	 */
	public static Certificate getCertificate (Class<?> clazz, String certificateType) throws CertificateException, NotFoundException, IOException, CertificateException {
		CertificateFactory certificateFactory = CertificateFactory.getInstance(certificateType);
		InputStream inputStream = getCertificateInputStream(clazz);
		try {
			return certificateFactory.generateCertificate(inputStream);
		} finally {
			inputStream.close();
		}
	}

	/**
	 * Gets an InputStream for reading a resource corresponding to the given class. Search will be conducted consecutively in the following places (in the example, the class is com.foo.Bar) :
	 * <OL>
	 * <LI>For a resource named com.foo.Bar.certificate (= com/foo/Bar.certificate) on the classpath</LI>
	 * <LI>For a file named com.foo.Bar.certificate in the current user working dir</LI>
	 * <LI>For a file named com.foo.Bar.certificate on the user's home dir</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            The class of the object for which a certificate is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @return An inputStream through which the resource corresponding to the clazz class can be read
	 * @throws NotFoundException
	 */
	public static InputStream getCertificateInputStream (Class<?> clazz) throws NotFoundException {
		return MyResource.getResourceInputStream(clazz, getCertificateBaseName(clazz));
	}

	/**
	 * Gets a Certificate object of the default certificate type.
	 * 
	 * @param clazz
	 *            The class of the object for which a certificate is to be found. The name of the resource that will be searched for is the full packagename, followed by the SUFFIX constant.
	 * @return The Certificate object containing the contents of the certificate resource corresponding to the clazz class/object
	 * @throws CertificateException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws CertificateException
	 */
	public static Certificate getPackageCertificate (Class<?> clazz) throws CertificateException, NotFoundException, IOException, CertificateException {
		return getPackageCertificate(clazz, MyCertificate.X509CERTIFICATETYPE);
	}

	/**
	 * Gets a Certificate object of the given certificate type.
	 * 
	 * @param clazz
	 *            The class of the object for which a certificate is to be found. The name of the resource that will be searched for is the full packagename, followed by the SUFFIX constant.
	 * @param certificateType
	 *            The name of the Certificate type to be returned to the caller
	 * @return The Certificate object containing the contents of the certificate resource corresponding to the clazz class/object
	 * @throws CertificateException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws CertificateException
	 */
	public static Certificate getPackageCertificate (Class<?> clazz, String certificateType) throws CertificateException, NotFoundException, IOException, CertificateException {
		CertificateFactory certificateFactory = CertificateFactory.getInstance(certificateType);
		InputStream inputStream = getPackageCertificateInputStream(clazz);
		try {
			return certificateFactory.generateCertificate(inputStream);
		} finally {
			inputStream.close();
		}
	}

	/**
	 * Gets an InputStream for reading a resource corresponding to the package of the given class. Search will be conducted consecutively in the following places (in the example, the class is com.foo.Bar) :
	 * <OL>
	 * <LI>For a resource named com.foo.certificate (= /com/foo/certificate) on the classpath</LI>
	 * <LI>For a file named com.foo.certificate in the current user working dir</LI>
	 * <LI>For a file named com.foo.certificate on the user's home dir</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            The class of the object for which a certificate is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @return An inputStream through which the resource corresponding to the clazz class can be read
	 * @throws NotFoundException
	 */
	public static InputStream getPackageCertificateInputStream (Class<?> clazz) throws NotFoundException {
		try {
			return MyResource.getResourceInputStream(clazz, RESOURCETYPE);
		} catch (NotFoundException e1) {
//			System.out.println("  packageCertificateInputStream failed to find " + RESOURCETYPE);
		}

		throw new NotFoundException();
	}

	/**
	 * Prevent instantiation
	 * 
	 */
	private MyCertificate ( ) {

	}
}
