/*
 * Created on 25-apr-2007
 */
package be.SIRAPRISE.util;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.UTFDataFormatException;

import be.erwinsmout.MyMessageFormat;

/**
 * @author Erwin Smout
 */
public class MyDataInputStream {

	/**
	 * Reads from the stream <code>in</code> a representation of a Unicode character string encoded in Java modified UTF-8 format; this string of characters is then returned as a <code>String</code>. The details of the modified UTF-8 representation are exactly the same as for the <code>readUTF</code> method of <code>DataInput</code>.
	 * 
	 * @param in
	 *            a data input stream.
	 * @return the string as decoded from the Unicode-encoded bytestream.
	 * @throws IOException
	 */
	public static String getBigUTFString (DataInputStream in) throws IOException {
		int charLength = in.readInt();
		int byteLength = in.readInt();
		StringBuffer str = new StringBuffer(charLength);
		byte bytearr[] = new byte[byteLength];
		int c, char2, char3;
		int count = 0;

		in.readFully(bytearr, 0, byteLength);

		while (count < byteLength) {
			c = bytearr[count] & 0xff;
			switch (c >> 4) {
				case 0:
				case 1:
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
					/* 0xxxxxxx */
					count++;
					str.append((char) c);
					break;
				case 12:
				case 13:
					/* 110x xxxx 10xx xxxx */
					count += 2;
					if (count > byteLength) {
						throw new UTFDataFormatException();
					}
					char2 = bytearr[count - 1];
					if ((char2 & 0xC0) != 0x80) {
						throw new UTFDataFormatException();
					}
					char appendChar = (char) (((c & 0x1F) << 6) | (char2 & 0x3F));
					//					
					str.append(appendChar);
					break;
				case 14:
					/* 1110 xxxx 10xx xxxx 10xx xxxx */
					count += 3;
					if (count > byteLength) {
						throw new UTFDataFormatException();
					}
					char2 = bytearr[count - 2];
					char3 = bytearr[count - 1];
					if (((char2 & 0xC0) != 0x80) || ((char3 & 0xC0) != 0x80)) {
						throw new UTFDataFormatException();
					}
					appendChar = (char) (((c & 0x0F) << 12) | ((char2 & 0x3F) << 6) | ((char3 & 0x3F) << 0));
					//					
					str.append(appendChar);
					break;
				default:
					/* 10xx xxxx, 1111 xxxx */
					throw new UTFDataFormatException();
			}
		}
		// The number of chars produced may be less than utflen
		return new String(str);
	}

	/**
	 * Reads from the stream <code>in</code> a representation of a Unicode character string encoded in Java modified UTF-8 format; this string of characters is then returned as a <code>String</code>. The details of the modified UTF-8 representation are exactly the same as for the <code>readUTF</code> method of <code>DataInput</code>.
	 * 
	 * @param in
	 *            a data input stream.
	 * @return the string as decoded from the Unicode-encoded bytestream.
	 * @throws IOException
	 * @throws UTFDataFormatException
	 */
	public static String getSmallUTFString (DataInputStream in) throws IOException, UTFDataFormatException {
		short charLength = in.readShort();
		short byteLength = in.readShort();
		String clientSignatureProtocolName = getUTFString(in, charLength, byteLength);
		return clientSignatureProtocolName;
	}

	/**
	 * Gets The String whose UTF-encoded bytes are the next bytes read from the inputStream argument
	 * 
	 * @param in
	 *            The InputStream to read
	 * @param charlength
	 *            The character length of the String
	 * @param byteLength
	 *            The number of bytes that are to be read from the inputstream
	 * @return The String whose UTF-encoded bytes are the next bytes read from the inputStream argument
	 * @throws IOException
	 * @throws UTFDataFormatException
	 */
	public static String getUTFString (DataInputStream in, int charlength, int byteLength) throws IOException, UTFDataFormatException {
		char[] chars = new char[charlength];
		int c = 0, bytes = 0;
		while (c < charlength && bytes < byteLength) {
			int b = in.readUnsignedByte();
			bytes++;
			char rc;
			switch (b >> 4) {
				case 0:
				case 1:
				case 2:
				case 3:
				case 4:
				case 5:
				case 6:
				case 7:
					/* 0xxxxxxx */
					rc = (char) b;
					break;
				case 12:
				case 13:
					/* 110x xxxx 10xx xxxx */
				{
					int b2 = in.readUnsignedByte();
					bytes++;
					if ((b2 & 0xC0) != 0x80)
						throw new UTFDataFormatException(MyMessageFormat.format("Received UTF byte 2 {0}.", new String[] { Integer.toHexString(b2) })); //$NON-NLS-1$
					rc = (char) (((b & 0x1F) << 6) | (b2 & 0x3F));
					break;
				}
				case 14:
					/* 1110 xxxx 10xx xxxx 10xx xxxx */
				{
					int b2 = in.readUnsignedByte();
					bytes++;
					int b3 = in.readUnsignedByte();
					bytes++;
					if (((b2 & 0xC0) != 0x80) || ((b3 & 0xC0) != 0x80))
						throw new UTFDataFormatException(MyMessageFormat.format("Received UTF bytes 2-3 {0}-{1}.", new String[] { Integer.toHexString(b2), Integer.toHexString(b3) })); //$NON-NLS-1$
					rc = ((char) (((b & 0x0F) << 12) | ((b2 & 0x3F) << 6) | ((b3 & 0x3F) << 0)));
					break;
				}
				default:
					/* 10xx xxxx, 1111 xxxx */
					throw new UTFDataFormatException(MyMessageFormat.format("First UTF byte {0} received.", new String[] { Integer.toHexString(b) })); //$NON-NLS-1$
			}
			chars[c++] = rc;
		}

		if (!(c == charlength && bytes == byteLength)) {
			throw new UTFDataFormatException(MyMessageFormat.format("Specified byte length {0} and char length {1} did not match with UTF data found.  Processed {2} bytes and found {3} chars.", new String[] { Integer.toString(byteLength), Integer.toString(charlength), Integer.toString(bytes), Integer.toString(c) })); //$NON-NLS-1$
		}
		return new String(chars);
	}

	/**
	 * Gets A byte array the size of byteCount holding the next bytes read from the inputStream argument
	 * 
	 * @param in
	 *            The InputStream to read
	 * @param byteCount
	 *            the number of bytes to read
	 * @return A byte array the size of byteCount holding the next bytes read from the inputStream argument
	 * @throws IOException
	 */
	public static byte[] readExactNumberOfBytes (DataInputStream in, int byteCount) throws IOException {
		byte[] inputBytes = new byte[byteCount];
		int totalBytesRead = 0;
		while (totalBytesRead != byteCount) {
			int bytesRead = in.read(inputBytes, totalBytesRead, byteCount - totalBytesRead);
			if (bytesRead < 0) {
				// premature EOF
				throw new PrematureEOFException(totalBytesRead, byteCount);
			}
			totalBytesRead += bytesRead;
		}
		return inputBytes;
	}
}