/**
 * 
 */
package be.SIRAPRISE.util;

import java.io.IOException;

import java.io.InputStream;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;

import be.SIRAPRISE.client.NAMES;
import be.erwinsmout.MyResource;
import be.erwinsmout.NotFoundException;

/**
 * @author Erwin
 * 
 */
public abstract class MyKeyStore {

	/**
	 * 
	 */
	public static final String RESOURCETPYE = "keystore"; //$NON-NLS-1$

	/**
	 * 
	 */
	public static final String SUFFIX = ".keystore"; //$NON-NLS-1$

	/**
	 * Gets the base name for the keystore resource corresponding to the given class. This is as follows (first rule that applies) :
	 * <OL>
	 * <LI>full classname does not end with 'KEYSTORE' (case insensitive) ===> full classname, with either of dots and forward slashes as separators, suffixed with '.KEYSTORE' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * <LI>full classname ends with '.KEYSTORE', or is equal to 'KEYSTORE' (both case insensitive) ===> full classname, with either of dots and forward slashes as separators (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties", or just "properties")</LI>
	 * <LI>full classname ends with 'KEYSTORE', not preceded by a dot (i.e. X.Y.ZKEYSTORE) ===> X.Y.Z, with either of dots and forward slashes as separators, suffixed with '.KEYSTORE' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            the class for which the corresponding properties resource name is to be returned
	 * @return the base name for the properties resource corresponding to the given class.
	 */
	protected static String getKeystoreBaseName (Class<?> clazz) {
		return MyResource.getResourceBaseName(clazz, RESOURCETPYE);
	}

	/**
	 * Gets the base name for the keystore resource corresponding to the given class. This is as follows (first rule that applies) :
	 * <OL>
	 * <LI>full classname does not end with 'KEYSTORE' (case insensitive) ===> full classname, with either of dots and forward slashes as separators, suffixed with '.KEYSTORE' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * <LI>full classname ends with '.KEYSTORE', or is equal to 'KEYSTORE' (both case insensitive) ===> full classname, with either of dots and forward slashes as separators (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties", or just "properties")</LI>
	 * <LI>full classname ends with 'KEYSTORE', not preceded by a dot (i.e. X.Y.ZKEYSTORE) ===> X.Y.Z, with either of dots and forward slashes as separators, suffixed with '.KEYSTORE' (e.g. "com.foo.Bar.properties" and "com/foo/Bar.properties")</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            the class for which the corresponding properties resource name is to be returned
	 * @return the base name for the properties resource corresponding to the given class.
	 */
	protected static String getPackageKeystoreBaseName (Class<?> clazz) {
		return MyResource.getPackageResourceBaseName(clazz, RESOURCETPYE);
	}

	/**
	 * Gets a KeyStore object of the default keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getKeyStore (Class<?> clazz) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		return getKeyStore(clazz, NAMES.DEFAULTKEYSTORETYPE);
	}

	/**
	 * Gets a KeyStore object of the default keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @param storePassword
	 *            The password for loading the Store
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getKeyStore (Class<?> clazz, char[] storePassword) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		return getKeyStore(clazz, NAMES.DEFAULTKEYSTORETYPE, storePassword);
	}

	/**
	 * Gets a KeyStore object of the given keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @param keyStoreType
	 *            The name of the KeyStore type to be returned to the caller
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getKeyStore (Class<?> clazz, String keyStoreType) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		return getKeyStore(clazz, keyStoreType, clazz.getSimpleName().toCharArray());
	}

	/**
	 * Gets a KeyStore object of the given keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @param keyStoreType
	 *            The name of the KeyStore type to be returned to the caller
	 * @param storePassword
	 *            The password for loading the store
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getKeyStore (Class<?> clazz, String keyStoreType, char[] storePassword) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		KeyStore jks = KeyStore.getInstance(keyStoreType);
		InputStream inputStream = getKeyStoreInputStream(clazz);
		try {
			jks.load(inputStream, storePassword);
		} finally {
			inputStream.close();
		}

		return jks;
	}

	/**
	 * Gets a keystore object for a keystore of the default keystore type for the named user.
	 * 
	 * @param user
	 *            The user whose keystore is to be obtained
	 * @return a keystore object for a keystore of the default keystore type for the named user.
	 * @throws KeyStoreException
	 * @throws IOException
	 * @throws CertificateException
	 * @throws NoSuchAlgorithmException
	 * @throws NotFoundException
	 */
	public static KeyStore getKeyStore (String user) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, NotFoundException {
		return getKeyStore(user, NAMES.DEFAULTKEYSTORETYPE);
	}

	/**
	 * Gets a keystore object for a keystore of the named keystore type for the named user.
	 * 
	 * @param user
	 *            The user whose keystore is to be obtained
	 * @param storePassword
	 *            the password to be provided for opening the keystore
	 * @return a keystore object for a keystore of the named keystore type for the named user.
	 * @throws KeyStoreException
	 * @throws IOException
	 * @throws CertificateException
	 * @throws NoSuchAlgorithmException
	 * @throws NotFoundException
	 */
	public static KeyStore getKeyStore (String user, char[] storePassword) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, NotFoundException {
		return getKeyStore(user, NAMES.DEFAULTKEYSTORETYPE, storePassword);
	}

	/**
	 * Gets a keystore object for a keystore of the named keystore type for the named user.
	 * 
	 * @param user
	 *            The user whose keystore is to be obtained
	 * @param keyStoreType
	 *            The keystore type
	 * @return a keystore object for a keystore of the named keystore type for the named user.
	 * @throws KeyStoreException
	 * @throws IOException
	 * @throws CertificateException
	 * @throws NoSuchAlgorithmException
	 * @throws NotFoundException
	 */
	public static KeyStore getKeyStore (String user, String keyStoreType) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, NotFoundException {
		return getKeyStore(user, keyStoreType, user.toCharArray());
	}

	/**
	 * Gets a keystore object for a keystore of the named keystore type for the named user.
	 * 
	 * @param user
	 *            The user whose keystore is to be obtained
	 * @param keyStoreType
	 *            The keystore type
	 * @param storePassword
	 *            the password to be provided for opening the keystore
	 * @return a keystore object for a keystore of the named keystore type for the named user.
	 * @throws KeyStoreException
	 * @throws IOException
	 * @throws CertificateException
	 * @throws NoSuchAlgorithmException
	 * @throws NotFoundException
	 *             If no keystore file for this user could be found
	 */
	public static KeyStore getKeyStore (String user, String keyStoreType, char[] storePassword) throws KeyStoreException, NoSuchAlgorithmException, CertificateException, IOException, NotFoundException {
		KeyStore jks = KeyStore.getInstance(keyStoreType);
		InputStream inputStream = getKeyStoreInputStream(user);
		try {
			jks.load(inputStream, storePassword);
		} finally {
			inputStream.close();
		}

		return jks;
	}

	/**
	 * Gets an InputStream for reading a resource corresponding to the given class. Search will be conducted consecutively in the following places (in the example, the class is com.foo.Bar) :
	 * <OL>
	 * <LI>For a resource named com/foo/Bar/keystore on the classpath</LI>
	 * <LI>For a file named com.foo.Bar.keystore in the current user working dir</LI>
	 * <LI>For a file named com.foo.Bar.keystore on the user's home dir</LI>
	 * <LI>For a resource named /com.foo.Bar.keystore on the classpath</LI>
	 * <LI>For a file named com/foo/Bar.keystore in the current user working dir</LI>
	 * <LI>For a file named com/foo/Bar.keystore on the user's home dir</LI>
	 * <LI>For a resource named /com/foo/Bar.keystore on the classpath</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @return An inputStream through which the resource corresponding to the clazz class can be read
	 * @throws NotFoundException
	 */
	public static InputStream getKeyStoreInputStream (Class<?> clazz) throws NotFoundException {
		return MyResource.getResourceInputStream(clazz, getKeystoreBaseName(clazz));
	}

	/**
	 * Gets an InputStream for reading a keystore resource corresponding to the named user. The search will be for a file named <user>.keystore in the directories pointed to by user.home and user.dir.
	 * 
	 * @param user
	 *            The user for which a keystore input stream is to be returned
	 * @return an InputStream for reading a keystore resource corresponding to the named user. The search will be for a file named <user>.keystore in the directories pointed to by user.home and user.dir.
	 * @throws NotFoundException
	 */
	public static InputStream getKeyStoreInputStream (String user) throws NotFoundException {
		return MyResource.getResourceInputStream(user);
	}

	/**
	 * Gets a KeyStore object of the default keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full packagename, followed by the SUFFIX constant.
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getPackageKeyStore (Class<?> clazz) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		return getPackageKeyStore(clazz, NAMES.DEFAULTKEYSTORETYPE);
	}

	/**
	 * Gets a KeyStore object of the default keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full packagename, followed by the SUFFIX constant.
	 * @param storePassword
	 *            The password for loading the store
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getPackageKeyStore (Class<?> clazz, char[] storePassword) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		return getPackageKeyStore(clazz, NAMES.DEFAULTKEYSTORETYPE, storePassword);
	}

	/**
	 * Gets a KeyStore object of the given keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full packagename, followed by the SUFFIX constant.
	 * @param keyStoreType
	 *            The name of the KeyStore type to be returned to the caller
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getPackageKeyStore (Class<?> clazz, String keyStoreType) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		return getPackageKeyStore(clazz, keyStoreType, clazz.getSimpleName().toCharArray());
	}

	/**
	 * Gets a KeyStore object of the given keystore type.
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full packagename, followed by the SUFFIX constant.
	 * @param keyStoreType
	 *            The name of the KeyStore type to be returned to the caller
	 * @param storePassword
	 *            The password for loading the store
	 * @return The KeyStore object containing the contents of the keystore resource corresponding to the clazz class/object
	 * @throws KeyStoreException
	 * @throws NotFoundException
	 * @throws IOException
	 * @throws NoSuchAlgorithmException
	 * @throws CertificateException
	 */
	public static KeyStore getPackageKeyStore (Class<?> clazz, String keyStoreType, char[] storePassword) throws KeyStoreException, NotFoundException, IOException, NoSuchAlgorithmException, CertificateException {
		KeyStore jks = KeyStore.getInstance(keyStoreType);
		InputStream inputStream = getPackageKeyStoreInputStream(clazz);
		try {
			jks.load(inputStream, storePassword);
		} finally {
			inputStream.close();
		}

		return jks;
	}

	/**
	 * Gets an InputStream for reading a resource corresponding to the package of the given class. Search will be conducted consecutively in the following places (in the example, the class is com.foo.Bar) :
	 * <OL>
	 * <LI>For a resource named com/foo/keystore on the classpath</LI>
	 * <LI>For a file named com.foo.keystore in the current user working dir</LI>
	 * <LI>For a file named com.foo.keystore on the user's home dir</LI>
	 * <LI>For a resource named /com.foo.keystore on the classpath</LI>
	 * <LI>For a file named com/foo.keystore in the current user working dir</LI>
	 * <LI>For a file named com/foo.keystore on the user's home dir</LI>
	 * <LI>For a resource named /com/foo.keystore on the classpath</LI>
	 * </OL>
	 * 
	 * @param clazz
	 *            The class of the object for which a keystore is to be found. The name of the resource that will be searched for is the full classname, followed by the SUFFIX constant.
	 * @return An inputStream through which the resource corresponding to the clazz class can be read
	 * @throws NotFoundException
	 */
	public static InputStream getPackageKeyStoreInputStream (Class<?> clazz) throws NotFoundException {
		return MyResource.getResourceInputStream(clazz, getPackageKeystoreBaseName(clazz));
	}

	/**
	 * Prevent instantiation
	 * 
	 */
	private MyKeyStore ( ) {

	}
}
