/*
 * Created on 28-mrt-2008
 */
package be.SIRAPRISE.util;

import java.util.*;
import java.util.Map.Entry;

import be.SIRAPRISE.client.AbstractRelation;
import be.SIRAPRISE.client.Heading;
import be.SIRAPRISE.client.AbstractTuple;
import be.SIRAPRISE.client.NAMES;
import be.SIRAPRISE.client.Tuple;
import be.SIRAPRISE.client.TypeDeclaration;
import be.SIRAPRISE.client.TypeIsScalarException;
import be.SIRAPRISE.client.NAMES.TYPENAMES;
import be.WAAR.PresentationLayer.*;
import be.WAAR.PresentationLayer.List;
import be.erwinsmout.NotFoundException;

/**
 * Utility class to produce WAAR list values from a SIRA_PRISE query response
 * 
 * @author Erwin Smout
 */
public class WaarListFactory {

	/**
	 * Creates a list value from a relation value obtained from a sira_prise server
	 * 
	 * @param listField
	 *            The listfield for which a value is being produced
	 * @param relationValue
	 *            The relation value in text form, starting with the RELATION type keyword. The HEADING part of the value may be present or may be omitted. I.e. the textual value may be RELATION(HEADING(...)BODY(...)) or it may be RELATION(BODY(...))
	 * @param userData
	 *            The userdata object
	 * @param locale
	 * @return A ListValue holding the information contained in the relationValue argument
	 * @throws WaarException
	 */
	public static ListValue fromSiraPriseRelationValue (PresentationField listField, String relationValue, UserData userData, Locale locale) throws WaarException {
		ListValue listValue = new ListValue(listField.getName());
		try {
			String val;
			if (relationValue.length() > 7 && relationValue.substring(0, 8).equalsIgnoreCase(TYPENAMES.RELATION)) {
				val = BracketParser.getNameValueFromEscaped(relationValue, 0).getNameValuePair().getValue();
			} else {
				val = relationValue.trim();
			}
			// val = HEADING(...)BODY(...) or BODY(...)

			HashMap<String, String> m0 = BracketParser.createMapFromEscaped(val);
			if (m0.containsKey(NAMES.POSSREPCOMPONENTNAMES.BODY)) {
				val = m0.get(NAMES.POSSREPCOMPONENTNAMES.BODY);
				// val = TUPLE(...)TUPLE(...)

				LinkedList<String> tuples = BracketParser.createListFromEscaped(val);
				Iterator<String> i_tuples = tuples.iterator();
				while (i_tuples.hasNext()) {
					HashMap<String, WaarValue> m = new HashMap<String, WaarValue>();
					String tupleValue = i_tuples.next();
					// tupleValue = anm1(av1)anm2(av2)...

					HashMap<String, String> am = BracketParser.createMapFromEscaped(tupleValue);
					Iterator<Entry<String, String>> i_am = am.entrySet().iterator();
					while (i_am.hasNext()) {
						Entry<String, String> me = i_am.next();
						String attributeName = me.getKey();
						String attributeValue = me.getValue();
						PresentationField listColumnField = listField.getColumn(attributeName, locale);
						if (listColumnField != null) {
							WaarValue listAttributeValue;
							if (listColumnField.getType() instanceof List) {
								listAttributeValue = fromSiraPriseRelationValue(listColumnField, attributeValue, userData, locale);
							} else {
								listAttributeValue = listColumnField.getType().encode(attributeValue, userData, locale);
							}
							m.put(attributeName, listAttributeValue);
						}
					}
					listValue.append(m);
				}

				return listValue;
			} else {
				throw new InvalidAttributeValueException(relationValue, "List", "Relation value specification does not contain a BODY()", locale); //$NON-NLS-1$//$NON-NLS-2$
			}
		} catch (NoClosingBracketException e) {
			throw new InvalidAttributeValueException(relationValue, "List", "Relation value specification does not contain a BODY()", locale); //$NON-NLS-1$//$NON-NLS-2$
		} catch (DuplicateNameException e) {
			throw new InvalidAttributeValueException(relationValue, "List", "A duplicate name was encountered in the response.", locale); //$NON-NLS-1$//$NON-NLS-2$
		} catch (NoOpeningBracketException e) {
			throw new InvalidAttributeValueException(relationValue, "List", "Relation value specification does not contain a BODY()", locale); //$NON-NLS-1$//$NON-NLS-2$
		} catch (InvalidEscapedCharacterException e1) {
			throw new RuntimeException(e1);
		} catch (MissingEscapedCharacterException e1) {
			throw new RuntimeException(e1);
		}
	}

	/**
	 * @param listFieldName
	 * @param rvaHeadingType
	 * @param bodyContent
	 * @param locale
	 * @return A ListValue holding the information contained in the relationValue argument
	 * @throws WaarException
	 */
	private static DynamicColumnListValue varColumnListFromSiraPriseRelationBody (String listFieldName, Heading rvaHeadingType, String bodyContent, Locale locale) throws WaarException {
		try {
			// HashMap headingAttributes = BracketParser.createMapFromNonEscaped(headingContent);

			DynamicColumnListValue listValue = new DynamicColumnListValue(listFieldName);
			Set<String> columns = rvaHeadingType.getAttributeNames();
			listValue.addColumns(columns);

			LinkedList<String> tuples = BracketParser.createListFromEscaped(bodyContent);
			Iterator<String> i_tuples = tuples.iterator();
			while (i_tuples.hasNext()) {
				HashMap<String, WaarValue> m = new HashMap<String, WaarValue>();
				String tupleValue = i_tuples.next();
				HashMap<String, String> attributeValues = BracketParser.createMapFromEscaped(tupleValue);
				Iterator<Entry<String, TypeDeclaration>> i_attributes = rvaHeadingType.entrySet().iterator();
				while (i_attributes.hasNext()) {
					Entry<String, TypeDeclaration> me = i_attributes.next();
					String attributeName = me.getKey();
					Object columnType = me.getValue();
					String attributeValue = attributeValues.get(attributeName);

					WaarValue listAttributeValue;
					if (columnType instanceof Heading) {
						// String innerHeadingTypeContent;
						// val = columnType;
						// innerHeadingTypeContent = BracketParser.getNameValueFromNonEscaped(columnType, 0).getNameValuePair().getValue();
						// val = innerHeadingTypeContent;
						// innerHeadingTypeContent = BracketParser.getNameValueFromNonEscaped(innerHeadingTypeContent, 0).getNameValuePair().getValue();
						Heading innerHeadingType;
						try {
							innerHeadingType = rvaHeadingType.getHeading(attributeName);
						} catch (NotFoundException impossible) {
							// assert false
							throw new RuntimeException(impossible);
						} catch (TypeIsScalarException impossible) {
							// assert false
							throw new RuntimeException(impossible);
						}
						String innerBodyContent;
						try {
							innerBodyContent = BracketParser.getNameValueFromEscaped(attributeValue, 0).getNameValuePair().getValue();
						} catch (InvalidEscapedCharacterException e1) {
							throw new RuntimeException(e1);
						} catch (MissingEscapedCharacterException e1) {
							throw new RuntimeException(e1);
						}
						listAttributeValue = varColumnListFromSiraPriseRelationBody(attributeName, innerHeadingType, innerBodyContent, locale);
					} else {
						listAttributeValue = new GenericStringValue(BracketParser.unMeta(attributeValue));
					}
					m.put(attributeName, listAttributeValue);
				}
				listValue.append(m);
			}
			return listValue;
		} catch (NoClosingBracketException e) {
			throw new RuntimeException(e);
		} catch (DuplicateNameException e) {
			throw new RuntimeException(e);
		} catch (NoOpeningBracketException e) {
			throw new RuntimeException(e);
		} catch (InvalidEscapedCharacterException e) {
			throw new RuntimeException(e);
		} catch (MissingEscapedCharacterException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Creates a ListValue off the textual result of a sira_prise query
	 * 
	 * @param listFieldName
	 *            The name of the dynamic-list field on the presentation
	 * @param relation
	 * @param locale
	 * @return A list value with columns as determined by the text value
	 * @throws WaarException
	 */
	public static DynamicColumnListValue varColumnListFromSiraPriseQueryResponse (String listFieldName, AbstractRelation relation, Locale locale) throws WaarException {
		DynamicColumnListValue listValue = new DynamicColumnListValue(listFieldName);

		Heading heading = relation.getHeading();
		Iterator<?> i = heading.entrySet().iterator();
		while (i.hasNext()) {
			Entry<?, ?> me = (Entry<?, ?>) i.next();
			String attributeName = (String) me.getKey();
			listValue.addColumn(attributeName);
		}
		// Set columns = heading.entrySetAttributeTypeNameMapping().keySet();
		// listValue.addColumns(columns);

		Iterator<Tuple> i_tuples = relation.iterator();
		while (i_tuples.hasNext()) {
			HashMap<String, WaarValue> m = new HashMap<String, WaarValue>();
			AbstractTuple tuple = (AbstractTuple) i_tuples.next();
			Map<?, ?> tupleAttributeValues = tuple.values();
			Iterator<?> i_tupleAttributeValues = tupleAttributeValues.entrySet().iterator();
			while (i_tupleAttributeValues.hasNext()) {
				Map.Entry<?, ?> me = (Entry<?, ?>) i_tupleAttributeValues.next();
				String attributeName = (String) me.getKey();
				String attributeValue = (String) me.getValue();

				String columnType;
				try {
					columnType = heading.getTypeName(attributeName);
				} catch (NotFoundException impossible) {
					// assert false
					throw new RuntimeException(impossible);
				}
				WaarValue listAttributeValue;
				if (columnType.toUpperCase().startsWith(TYPENAMES.RELATION)) {
					Heading rvaHeadingType;
					try {
						rvaHeadingType = heading.getHeading(attributeName);
					} catch (NotFoundException impossible) {
						// assert false
						throw new RuntimeException(impossible);
					} catch (TypeIsScalarException impossible) {
						// assert false
						throw new RuntimeException(impossible);
					}
					// String innerHeadingTypeContent;
					// try {
					// innerHeadingTypeContent = BracketParser.getNameValueFromNonEscaped(columnType, 0).getNameValuePair().getValue(); //HEADING(...)
					// innerHeadingTypeContent = BracketParser.getNameValueFromNonEscaped(innerHeadingTypeContent, 0).getNameValuePair().getValue(); //ATTR(TYPE)ATTR(TYPE)
					// } catch (NoClosingBracketException e2) {
					// throw new ExpectedBracketsMissingException(columnType);
					// }
					String innerBodyContent;
					try {
						innerBodyContent = BracketParser.getNameValueFromEscaped(attributeValue, 0).getNameValuePair().getValue(); // BODY(...)
					} catch (NoClosingBracketException e3) {
						throw new RuntimeException(e3);
					} catch (InvalidEscapedCharacterException e) {
						throw new RuntimeException(e);
					} catch (MissingEscapedCharacterException e) {
						throw new RuntimeException(e);
					}
					listAttributeValue = varColumnListFromSiraPriseRelationBody(attributeName, rvaHeadingType, innerBodyContent, locale);
				} else {
					try {
						listAttributeValue = new GenericStringValue(BracketParser.unMeta(attributeValue));
					} catch (InvalidEscapedCharacterException e) {
						throw new RuntimeException(e);
					} catch (MissingEscapedCharacterException e) {
						throw new RuntimeException(e);
					}
				}
				m.put(attributeName, listAttributeValue);
			}
			listValue.append(m);
		}

		return listValue;
	}
}